<?php
/**
 * Theme functions and definitions.
 *
 * @link https://codex.wordpress.org/Functions_File_Explained
 *
 * @package tour_travel_package
 */

if ( ! function_exists( 'tour_travel_package_setup' ) ) :
	/**
	 * Sets up theme defaults and registers support for various WordPress features.
	 *
	 * Note that this function is hooked into the after_setup_theme hook, which
	 * runs before the init hook. The init hook is too late for some features, such
	 * as indicating support for post thumbnails.
	 */
	function tour_travel_package_setup() {
		/*
		 * Make theme available for translation.
		 */ 
		load_theme_textdomain( 'tour-travel-package', get_template_directory() . '/languages' );

		// Add default posts and comments RSS feed links to head.
		add_theme_support( 'automatic-feed-links' );

		// Let WordPress manage the document title.
		add_theme_support( 'title-tag' );

		// Enable support for Post Thumbnails on posts and pages.
		add_theme_support( 'post-thumbnails' );
		add_image_size( 'tour-travel-package-thumb', 400, 300 );

		// Register nav menu locations.
		register_nav_menus( array(
			'primary-menu'  => esc_html__( 'Primary Menu', 'tour-travel-package' ),
		) );

		/*
		* This theme styles the visual editor to resemble the theme style,
		* specifically font, colors, icons, and column width.
		*/
		$min = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
		add_editor_style( array( '/css/editor-style' . $min . '.css', tour_travel_package_fonts_url() ) );

		/*
		 * Switch default core markup to output valid HTML5.
		 */
		add_theme_support( 'html5', array(
			'comment-form',
			'comment-list',
			'gallery',
			'caption',
		) );

		// Set up the WordPress core custom background feature.
		add_theme_support( 'custom-background', apply_filters( 'tour_travel_package_custom_background_args', array(
			'default-color' => 'f7fcfe',
		) ) );

		// Enable support for selective refresh of widgets in Customizer.
		add_theme_support( 'customize-selective-refresh-widgets' );

		// Enable support for custom logo.
		add_theme_support( 'custom-logo' );

		// Load default block styles.
		add_theme_support( 'wp-block-styles' );

		// Add support for responsive embeds.
		add_theme_support( 'responsive-embeds' );

		// Enable support for footer widgets.
		add_theme_support( 'footer-widgets', 4 );

		// Load Supports.
		require_once get_template_directory() . '/inc/support.php';

		// Add custom editor font sizes.
		add_theme_support(
			'editor-font-sizes',
			array(
				array(
					'name'      => __( 'Small', 'tour-travel-package' ),
					'shortName' => __( 'S', 'tour-travel-package' ),
					'size'      => 13,
					'slug'      => 'small',
				),
				array(
					'name'      => __( 'Normal', 'tour-travel-package' ),
					'shortName' => __( 'M', 'tour-travel-package' ),
					'size'      => 14,
					'slug'      => 'normal',
				),
				array(
					'name'      => __( 'Large', 'tour-travel-package' ),
					'shortName' => __( 'L', 'tour-travel-package' ),
					'size'      => 30,
					'slug'      => 'large',
				),
				array(
					'name'      => __( 'Huge', 'tour-travel-package' ),
					'shortName' => __( 'XL', 'tour-travel-package' ),
					'size'      => 36,
					'slug'      => 'huge',
				),
			)
		);

		// Editor color palette.
		add_theme_support(
			'editor-color-palette',
			array(
				array(
					'name'  => __( 'Black', 'tour-travel-package' ),
					'slug'  => 'black',
					'color' => '#2f354d',
				),
				array(
					'name'  => __( 'White', 'tour-travel-package' ),
					'slug'  => 'white',
					'color' => '#ffffff',
				),
				array(
					'name'  => __( 'Gray', 'tour-travel-package' ),
					'slug'  => 'gray',
					'color' => '#727272',
				),
				array(
					'name'  => __( 'Blue', 'tour-travel-package' ),
					'slug'  => 'blue',
					'color' => '#f0923c',
				),
				array(
					'name'  => __( 'Navy Blue', 'tour-travel-package' ),
					'slug'  => 'navy-blue',
					'color' => '#f0923c',
				),
				array(
					'name'  => __( 'Light Blue', 'tour-travel-package' ),
					'slug'  => 'light-blue',
					'color' => '#f7fcfe',
				),
				array(
					'name'  => __( 'Orange', 'tour-travel-package' ),
					'slug'  => 'orange',
					'color' => '#121212',
				),
				array(
					'name'  => __( 'Green', 'tour-travel-package' ),
					'slug'  => 'green',
					'color' => '#77a464',
				),
				array(
					'name'  => __( 'Red', 'tour-travel-package' ),
					'slug'  => 'red',
					'color' => '#e4572e',
				),
				array(
					'name'  => __( 'Yellow', 'tour-travel-package' ),
					'slug'  => 'yellow',
					'color' => '#f4a024',
				),
			)
		);

		global $pagenow;
		if (is_admin() && ('themes.php' == $pagenow) && isset($_GET['activated']) && wp_get_theme()->get('TextDomain') === 'tour-travel-package') {
			wp_redirect(admin_url('themes.php?page=tour-travel-package-getting-started'));
		}
	}
endif;

add_action( 'after_setup_theme', 'tour_travel_package_setup' );

/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 */
function tour_travel_package_content_width() {
	$GLOBALS['content_width'] = apply_filters( 'tour_travel_package_content_width', 771 );
}
add_action( 'after_setup_theme', 'tour_travel_package_content_width', 0 );

/**
 * Register widget area.
 */
function tour_travel_package_widgets_init() {
	register_sidebar( array(
		'name'          => esc_html__( 'Primary Sidebar', 'tour-travel-package' ),
		'id'            => 'sidebar-1',
		'description'   => esc_html__( 'Add widgets here to appear in your Primary Sidebar.', 'tour-travel-package' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );
	register_sidebar( array(
		'name'          => esc_html__( 'Secondary Sidebar', 'tour-travel-package' ),
		'id'            => 'sidebar-2',
		'description'   => esc_html__( 'Add widgets here to appear in your Secondary Sidebar.', 'tour-travel-package' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );

	register_sidebar( array(
		'name'          => esc_html__( 'Secondary Sidebar 1', 'tour-travel-package' ),
		'id'            => 'sidebar-3',
		'description'   => esc_html__( 'Add widgets here to appear in your Secondary Sidebar 1.', 'tour-travel-package' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );
}
add_action( 'widgets_init', 'tour_travel_package_widgets_init' );

/**
 * Register custom fonts.
 */
function tour_travel_package_fonts_url() {
	$font_family   = array(
		'Lexend:wght@100;200;300;400;500;600;700;800;900',
	);
	
	$fonts_url = add_query_arg( array(
		'family' => implode( '&family=', $font_family ),
		'display' => 'swap',
	), 'https://fonts.googleapis.com/css2' );

	$contents = wptt_get_webfont_url( esc_url_raw( $fonts_url ) );
	return $contents;
}

/**
 * Enqueue scripts and styles.
 */
function tour_travel_package_scripts() {

	$min = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';

	wp_enqueue_style( 'tour-travel-package-font-awesome', get_template_directory_uri() . '/third-party/font-awesome/css/font-awesome' . $min . '.css', '', '4.7.0' );

	$fonts_url = tour_travel_package_fonts_url();
	if ( ! empty( $fonts_url ) ) {
		wp_enqueue_style( 'tour-travel-package-google-fonts', $fonts_url, array(), null );
	}

	wp_enqueue_style( 'dashicons' );

	wp_enqueue_style('bootstrap-style', get_template_directory_uri().'/css/bootstrap.css');

	// Theme stylesheet.
	wp_enqueue_style( 'tour-travel-package-style', get_stylesheet_uri(), null, date( 'Ymd-Gis', filemtime( get_template_directory() . '/style.css' ) ) );

	wp_enqueue_style( 'tour-travel-package-style', get_stylesheet_uri() );
	wp_style_add_data( 'tour-travel-package-style', 'rtl', 'replace' );

	// Theme block stylesheet.
	wp_enqueue_style( 'tour-travel-package-block-style', get_theme_file_uri( '/css/blocks.css' ), array( 'tour-travel-package-style' ), '20211006' );
	
	wp_enqueue_script( 'tour-travel-package-custom-js', get_template_directory_uri(). '/js/custom.js', array('jquery') ,'',true);
	
	wp_enqueue_script( 'jquery-superfish', get_theme_file_uri( '/js/jquery.superfish.js' ), array( 'jquery' ), '2.1.2', true );
	wp_enqueue_script('bootstrap-js', get_template_directory_uri().'/js/bootstrap.js', array('jquery'), '', true);
	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}
}
add_action( 'wp_enqueue_scripts', 'tour_travel_package_scripts' );

/**
 * Enqueue styles for the block-based editor.
 *
 * @since Tour Travel Package
 */
function tour_travel_package_block_editor_styles() {
	// Theme block stylesheet.
	wp_enqueue_style( 'tour-travel-package-editor-style', get_template_directory_uri() . '/css/editor-blocks.css', array(), '20101208' );

	$fonts_url = tour_travel_package_fonts_url();
	if ( ! empty( $fonts_url ) ) {
		wp_enqueue_style( 'tour-travel-package-google-fonts', $fonts_url, array(), null );
	}
}
add_action( 'enqueue_block_editor_assets', 'tour_travel_package_block_editor_styles' );

define( 'IS_FREE_MIZAN_THEMES', 'tour-travel-package' );

/**
 * Load init.
 */
require_once get_template_directory() . '/inc/init.php';

/**
 *  Webfonts 
 */
require_once get_template_directory() . '/inc/wptt-webfont-loader.php';

require_once get_template_directory() . '/inc/recommendations/tgm.php';

require_once get_template_directory() . '/inc/upsell/class-upgrade-pro.php';

require get_template_directory() . '/inc/getting-started/getting-started.php';

require get_template_directory() . '/inc/getting-started/plugin-activation.php';

define('TOUR_TRAVEL_PACKAGE_SUPPORT',__('https://wordpress.org/support/theme/tour-travel-package/','tour-travel-package'));
define('TOUR_TRAVEL_PACKAGE_REVIEW',__('https://wordpress.org/support/theme/tour-travel-package/reviews/','tour-travel-package'));
define('TOUR_TRAVEL_PACKAGE_BUY_NOW',__('https://www.mizanthemes.com/elementor/tour-package-wordpress-theme/','tour-travel-package'));
define('TOUR_TRAVEL_PACKAGE_LIVE_DEMO',__('https://mizanthemes.com/preview/tour-travel-package/','tour-travel-package'));
define('TOUR_TRAVEL_PACKAGE_PRO_DOC',__('https://www.mizanthemes.com/setup-guide/tour-travel-package-pro/','tour-travel-package'));